<?php
/**
 * Update Cache Clearer
 * Adds a "Clear Update Cache" button for admins when updates might be stuck
 * 
 * @package WooCommerce_MyParcel
 * @since 5.2.1
 */

if (!defined('ABSPATH')) exit;

class WC_MyParcel_Update_Cache_Clearer {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_notices', array($this, 'show_cache_clear_notice'));
        add_action('admin_post_myparcel_clear_update_cache', array($this, 'clear_update_cache'));
    }
    
    /**
     * Show admin notice with clear cache button
     */
    public function show_cache_clear_notice() {
        // Only show on plugins page or updates page
        $screen = get_current_screen();
        if (!$screen || !in_array($screen->id, array('plugins', 'update-core', 'dashboard'))) {
            return;
        }
        
        // Only show to admins
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Check if we just cleared cache
        if (isset($_GET['cache_cleared']) && $_GET['cache_cleared'] === '1') {
            ?>
            <div class="notice notice-success is-dismissible">
                <p>
                    <strong>✅ Update cache geleegd!</strong> 
                    WordPress zal nu opnieuw checken voor updates. 
                    Klik op "Check opnieuw" als je geen update melding ziet.
                </p>
            </div>
            <?php
            return;
        }
        
        // Show the clear cache button
        $clear_url = admin_url('admin-post.php?action=myparcel_clear_update_cache');
        $clear_url = wp_nonce_url($clear_url, 'myparcel_clear_cache');
        
        ?>
        <div class="notice notice-info" style="display: flex; align-items: center; padding: 12px;">
            <div style="flex: 1;">
                <strong>🔄 MyParcel Updates</strong><br>
                <span style="font-size: 13px;">
                    Zie je geen nieuwe update? Probeer de update cache te legen.
                </span>
            </div>
            <div>
                <a href="<?php echo esc_url($clear_url); ?>" 
                   class="button button-primary"
                   onclick="return confirm('Update cache legen en opnieuw checken voor updates?');">
                    🗑️ Clear Update Cache
                </a>
            </div>
        </div>
        <?php
    }
    
    /**
     * Clear update cache
     */
    public function clear_update_cache() {
        // Verify nonce
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce($_GET['_wpnonce'], 'myparcel_clear_cache')) {
            wp_die('Security check failed');
        }
        
        // Verify permissions
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        // Clear WordPress update transients
        delete_site_transient('update_plugins');
        delete_site_transient('update_themes');
        delete_site_transient('update_core');
        
        delete_transient('update_plugins');
        delete_transient('update_themes');
        
        // Clear plugin-specific transients
        delete_transient('dyronics_update_check_woocommerce-myparcel');
        delete_transient('dyronics_license_check_woocommerce-myparcel');
        delete_transient('dyronics_last_update_check');
        delete_site_transient('dyronics_update_check_woocommerce-myparcel');
        
        // Clear any cached plugin data
        wp_cache_delete('plugins', 'plugins');
        wp_cache_flush();
        
        // Force WordPress to check for plugin updates
        wp_clean_plugins_cache();
        wp_update_plugins();
        
        // Log the action
        if (class_exists('WC_MyParcel_Logger')) {
            WC_MyParcel_Logger::log(
                'Update cache cleared by ' . wp_get_current_user()->user_login,
                'info',
                'system'
            );
        }
        
        // Redirect back with success message
        $redirect_url = add_query_arg(
            array('cache_cleared' => '1'),
            admin_url('plugins.php')
        );
        
        wp_redirect($redirect_url);
        exit;
    }
}

// Initialize
new WC_MyParcel_Update_Cache_Clearer();
